<?php
/**
 * SkeletonCart
 * (c) 2017 Juxtaposition Inc. All Rights Reserved.
 * (c) 2017 Curlybracket, Inc. All Rights Reserved.
 * This code cannot be redistributed without permission from the copyright holder.
 * For more information, consult your SkeletonCart license.
 **/

namespace WeightPostage;

use App;
use App\Cart;
use App\Slip;

class Postage extends \App\Postage
{
    protected $fileName;
    protected $weightPostage = array();
    protected $categories = array();

    public function __construct($table, $freePostagePrice)
    {
        parent::__construct($table, $freePostagePrice);

        $this->fileName = __DIR__ . '/data/postage.csv';

        $this->readDataFile();
    }

    public function readDataFile()
    {
        $fp = false;

        if (($fp = fopen($this->fileName, 'r')) === false) {
            throw new \RuntimeException("{$this->fileName}が開けません。");
        }

        try {
            $categoriesCount = 0;
            $lineNo = 0;
            while (($line = fgetcsv($fp)) !== false) {
                if ($lineNo++ == 0) {
                    $categories = array_slice($line, 1);
                    if (($categoriesCount = count($categories)) === 0) {
                        throw new \RuntimeException('重量区分が設定されていません。');
                    }
                    $this->categories = $categories;
                    continue;
                }
                $prefId = array_shift($line);
                if ($prefId < 1 || $prefId > 47) {
                    throw new \RuntimeException('不正な都道府県コードが設定されています。');
                }
                $colCount = count($line);
                if ($colCount === 0 || $colCount !== $categoriesCount) {
                    throw new \RuntimeException("重量区分の数が一致しません: categories: $categoriesCount, cols: $colCount");
                }
                $this->weightPostage[$prefId] = $line;
            }

            $prefCount = count(array_keys($this->weightPostage));
            if ($prefCount != 47) {
                throw new \RuntimeException("全ての都道府県に送料が設定されていません, prefs: $prefCount, lineNo: $lineNo");
            }
        } catch (Exception $e) {
            // HACK: for php 5.3
        } /* finally */
        {
            if ($fp !== false) {
                fclose($fp);
            }
            if (isset($e)) {
                throw $e;
            }
        }
    }

    public function calculate(Cart $cart, Slip $slip)
    {
        if($this->isFreePostage($cart, $slip)) {
            return 0;
        }
        $prefId = $slip->getDestinationPrefecture();
        if (!isset($prefId) || $prefId < 1 || $prefId > 47) {
            return 0;
        }
        $weight = $slip->weight();
        $categoryIndex = $this->findCategoryIndex($weight);
        if ($categoryIndex === false || !isset($this->weightPostage[$prefId][$categoryIndex])) {
            throw new \RuntimeException('不正な送料設定です。');
        }
        $postage = $this->weightPostage[$prefId][$categoryIndex];
        return $postage;
    }

    public function findCategoryIndex($weight)
    {
        if (!isset($weight) || is_int($weight)) {
            throw new \UnexpectedValueException('重量が指定されていません。');
        }
        $categoryCount = count($this->categories);
        for ($index = 0; $index < $categoryCount; $index++) {
            if ($weight <= $this->categories[$index]) {
                return $index;
            }
        }
        return false;
    }
}
